/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var _1 = require("./");
var path_1 = require("path");
var defaultpath = require("path");
var fs = require("fs");
var NODE_MODULES_DIRNAME = "node_modules";
var PACKAGE_JSON_NAME = "package.json";
var VALID_NON_CODE_LOCATIONS = ["native", "NonError", "unknown"];
var StackProcessor = (function () {
    function StackProcessor(options) {
        this.doFileIO = true;
        this.appRoot = options.appRoot;
        this._errorReporter = options.errorReporter;
        this._packageInfo = options.packageInfo;
        switch (options.mode) {
            case _1.pathMode.defaultMode:
                this.path = defaultpath;
                break;
            case _1.pathMode.posix:
                this.path = path_1.posix;
                break;
            case _1.pathMode.windows:
                this.path = path_1.win32;
                break;
            default:
                this.path = defaultpath;
        }
    }
    StackProcessor.prototype.parseFirstLine = function (firstLine) {
        var firstLineRX = /(^.+?\b)(?:: )?(?:(.+?\b): )?(.*)/;
        var ERROR_SS = 1;
        var SYSTEM_ERROR_SS = 2;
        var MESSAGE_SS = 3;
        var matches = firstLine.match(firstLineRX);
        var type;
        if (matches[ERROR_SS] === "Error" && matches[SYSTEM_ERROR_SS]) {
            type = matches[SYSTEM_ERROR_SS];
        }
        else {
            type = matches[ERROR_SS];
        }
        var message = matches[MESSAGE_SS];
        return { ErrorType: type, ErrorMessage: message, Stack: [] };
    };
    StackProcessor.prototype.parseStackLine = function (inputLine, getPackageInfo) {
        if (getPackageInfo === void 0) { getPackageInfo = false; }
        var isEval = false;
        if (inputLine.includes("eval at")) {
            isEval = true;
            inputLine = this.cleanEvalFrames(inputLine);
        }
        inputLine = _1.replaceAll(inputLine, "(anonymous function)", "(anonymousfunction)");
        inputLine = _1.replaceAll(inputLine, "unknown location", "unknown");
        var mainRX = /\s*at (new )?(.+?)(?: \[as ([^ ]+?)\])?(?: \((.*)\))?$/;
        var newSS = 1;
        var functionNameSS = 2;
        var asMethodSS = 3;
        var locationSS = 4;
        var functionName;
        var location;
        var atNew = false;
        var asString = "";
        var mainMatches = inputLine.match(mainRX);
        if (!mainMatches) {
            var e = new Error("unrecognized stack trace format: " + inputLine);
            if (this._errorReporter) {
                this._errorReporter.emitInternalError(e, null, inputLine);
            }
            var temp = {
                RelativePath: "UNSUPPORTED",
                Package: { name: "UNSUPPORTED" },
                line: -1,
                column: -1,
                extra: { failedToParse: true } };
            return temp;
        }
        if (mainMatches[newSS]) {
            atNew = true;
        }
        if (mainMatches[asMethodSS]) {
            asString = mainMatches[asMethodSS];
        }
        if (mainMatches[locationSS]) {
            location = mainMatches[locationSS];
            functionName = mainMatches[functionNameSS];
        }
        else {
            location = mainMatches[functionNameSS];
            functionName = "";
        }
        var result = this.parseCodeLocation(location);
        result.FunctionName = functionName;
        var stripped = this.getPackageInfo(result.RelativePath);
        result.RelativePath = stripped.strippedPath;
        if (getPackageInfo && stripped.package) {
            result.Package = stripped.package;
        }
        if (atNew || asString || isEval) {
            if (atNew) {
                result.FunctionName += ".constructor";
            }
            if (asString) {
                result.FunctionName += "[" + asString + "]";
            }
            result.extra = { isNew: atNew, hasAs: asString, isEval: isEval, failedToParse: false, rawString: "" };
        }
        return result;
    };
    StackProcessor.prototype.cleanEvalFrames = function (input) {
        var evalRX = /\eval at .+\), /;
        return input.replace(evalRX, "");
    };
    StackProcessor.prototype.parseCodeLocation = function (input) {
        var i = VALID_NON_CODE_LOCATIONS.length;
        var element;
        while (i--) {
            element = VALID_NON_CODE_LOCATIONS[i];
            if (element === input) {
                return { RelativePath: input, line: -1, column: -1, FunctionName: "" };
            }
        }
        input = _1.replaceAll(input, "file:///", "");
        input = _1.replaceAll(input, "%20", " ");
        var locationRX = /(?:(^.+):(\d+):(\d+))/;
        var PATH_SS = 1;
        var LINE_SS = 2;
        var COLUMN_SS = 3;
        var line;
        var column;
        var matches = input.match(locationRX);
        if (!matches) {
            var intError = new Error("unsupported location format");
            if (this._errorReporter) {
                this._errorReporter.emitInternalError(intError, null, input);
            }
            return { RelativePath: "UNSUPPORTED", line: -1, column: -1, FunctionName: "" };
        }
        else {
            line = parseInt(matches[LINE_SS], 10);
            column = parseInt(matches[COLUMN_SS], 10);
        }
        var path = this.path.normalize(matches[PATH_SS]);
        return { RelativePath: path, line: line, column: column, FunctionName: "" };
    };
    StackProcessor.prototype.getPackageInfo = function (filePath) {
        var i = VALID_NON_CODE_LOCATIONS.length;
        var element;
        while (i--) {
            element = VALID_NON_CODE_LOCATIONS[i];
            if (element === filePath) {
                return { package: { name: "UNKNOWN" }, strippedPath: filePath };
            }
        }
        if (!this.path.isAbsolute(filePath)) {
            return {
                package: { name: "node", version: process.version },
                strippedPath: filePath };
        }
        var relPath = this.path.relative(this.appRoot, filePath);
        var nodeModIndex = filePath.lastIndexOf(NODE_MODULES_DIRNAME);
        if (nodeModIndex !== -1) {
            var modRoot = filePath.substring(0, nodeModIndex + NODE_MODULES_DIRNAME.length + 1);
            var modPath = this.path.relative(modRoot, filePath);
            var sepIndex = modPath.indexOf(this.path.sep);
            if (sepIndex !== -1) {
                var modName = modPath.substring(0, sepIndex);
                relPath = modPath.substring(sepIndex + 1);
                var pkg = { name: modName };
                if (this.doFileIO) {
                    try {
                        var fullPath = this.path.join(modRoot, modName, PACKAGE_JSON_NAME);
                        var pkgjs = JSON.parse(fs.readFileSync(fullPath, "utf8"));
                        pkg.packageJsonFullPath = fullPath;
                        if (pkgjs.name) {
                            pkg.name = pkgjs.name;
                        }
                        if (pkgjs.version) {
                            pkg.version = pkgjs.version;
                        }
                    }
                    catch (e) {
                    }
                }
                relPath = _1.convertPathToPosix(relPath);
                return { package: pkg, strippedPath: relPath };
            }
            if (this._errorReporter) {
                this._errorReporter.emitInternalError(new Error("javascript file in unexpected location"), null, filePath);
            }
            filePath = this.path.basename(filePath);
            return { package: { name: "UNKNOWN" }, strippedPath: filePath };
        }
        if (filePath.includes(".asar")) {
            var index = filePath.lastIndexOf(".asar") + ".asar".length;
            var asarName = this.path.basename(filePath.substring(0, index));
            var relativePath = _1.convertPathToPosix(filePath.substring(index + 1));
            relPath = _1.convertPathToPosix(relPath);
            return { package: { name: asarName }, strippedPath: relativePath };
        }
        if (relPath === filePath || relPath.startsWith("..") || this.appRoot === "") {
            filePath = this.path.basename(filePath);
            return { package: { name: "UNKNOWN" }, strippedPath: filePath };
        }
        relPath = _1.convertPathToPosix(relPath);
        return { package: this._packageInfo, strippedPath: relPath };
    };
    return StackProcessor;
}());
exports.StackProcessor = StackProcessor;
function removeUserName(input) {
    var userNameRX = /[a-z]:\\Users\\([^\\]+)/i;
    var match = input.match(userNameRX);
    if (match) {
        input = _1.replaceAll(input, match[1], "USERNAME");
    }
    return input;
}
exports.removeUserName = removeUserName;
function structuredToSemiWatson(input) {
    var _this = this;
    var WLB = _1.WATSON_LINE_BREAK;
    var result = "";
    input.Stack.forEach(function (frame) {
        result += (result ? WLB : "") + _this.frameToString(frame);
    });
    return result;
}
exports.structuredToSemiWatson = structuredToSemiWatson;
function structuredToStandard(input) {
    var stack = "";
    input.Stack.forEach(function (frame) {
        if (frame.FunctionName) {
            stack += (stack ? "\n" : "") + ("" + ((frame.extra && frame.extra.isEval) ? "eval " : ""))
                + ("at " + frame.FunctionName + " ") +
                ("(" + frame.RelativePath);
            if (frame.line !== -1 && frame.column !== -1) {
                stack += ":" + frame.line + ":" + frame.column;
            }
            stack += ")";
        }
        else {
            stack += (stack ? "\n" : "") + (((frame.extra && frame.extra.isEval) ? "eval " : "") + "at ") +
                ("" + frame.RelativePath);
            if (frame.line !== -1 && frame.column !== -1) {
                stack += ":" + frame.line + ":" + frame.column;
            }
        }
    });
    return stack;
}
exports.structuredToStandard = structuredToStandard;
function frameToString(frame) {
    if (!frame) {
        return "UNSUPPORTED\tUNSUPPORTED";
    }
    var packageString = frame.Package.name;
    if (!packageString) {
        packageString = "UNKNOWN";
    }
    if (frame.Package.version) {
        packageString += "@" + frame.Package.version;
    }
    var fileInfo = frame.RelativePath;
    if (!fileInfo) {
        fileInfo = "UNKNOWN";
    }
    if (frame.line !== -1 && frame.column !== -1) {
        fileInfo += ":" + frame.line + ":" + frame.column;
    }
    var result = packageString + "\t" + fileInfo;
    if (frame.FunctionName) {
        result += "\t" + frame.FunctionName;
    }
    return result;
}
exports.frameToString = frameToString;

// SIG // Begin signature block
// SIG // MIIh8QYJKoZIhvcNAQcCoIIh4jCCId4CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // AJ1T0BPhTpJWXqAHvQ1Fgisfsj4GjdxdYObJrXSUYgyg
// SIG // gguEMIIFDDCCA/SgAwIBAgITMwAAAU/nxmLJRvSpfwAA
// SIG // AAABTzANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTE2MTExNzIxNTkxNFoX
// SIG // DTE4MDIxNzIxNTkxNFowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBALSJkIp2DKzFPaG8gejXUAi7
// SIG // WmmnrsfMuFu9OA8hUUYII9G9Iplztyjv9werK/lI96UU
// SIG // BalNJBttYZacg50zwCx5mfBd8+zVRN+lzVn6dq0GP8ZK
// SIG // A+bqLZQQwmZv/WLQUIrRIpEjl2U0ocSaAl6S4rh8KHjC
// SIG // XuNG98rvOFeIrFYGWMqYG8eo1142rppFMOO2tz0Ywqcn
// SIG // sHlNU9gy5najUCr2P7q7QR4khk2Ym3qe9izivJBgZfuW
// SIG // afrgE2jiiSoudRidwm24IfSZXip/MaCnDczvTxFu/klC
// SIG // TmHeN8srnkDvzqtecAhzpXJ7t0bOWzJTXYo5UW+w16h4
// SIG // 3aMob17uQh8CAwEAAaOCAXswggF3MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjc9BgEGCCsGAQUFBwMDMB0GA1UdDgQWBBSS
// SIG // DqERcbYvVQ3DD/TJXHQuWgxqvzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwODY1
// SIG // K2I0YjEyODc4LWUyOTMtNDNlOS1iMjFlLTdkMzA3MTlk
// SIG // NDUyZjAfBgNVHSMEGDAWgBTm/F97uyIAWORyTrX0IXQj
// SIG // MubvrDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNDb2RTaWdQQ0FfMjAxMC0wNy0wNi5jcmwwWgYIKwYB
// SIG // BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNp
// SIG // Z1BDQV8yMDEwLTA3LTA2LmNydDAMBgNVHRMBAf8EAjAA
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQARwLlqQzX+hbE4gA24
// SIG // dHUsjXFJuDxB7S294gY2kHYrieW2gemG4xiMlpMyQqen
// SIG // 9rc0JTnQxnpD6tt3cMzU6ULrqMVVAnV4ntgQvgcbypRj
// SIG // +zwPce7UVWfXKfZqvwe85Y+C6vih4nMzPxSSXqG6ZVc3
// SIG // RYWAXNYsxos8aoejBxHL7NlxOCA1JD1UvcxVifR3EJvl
// SIG // maoPrsY8jp/aAE/PfeGzwCD3UjH3IkjVYY1/JgIZfYPM
// SIG // YZMq3bjc4gizp9/hBHdgyPSsNcsx5Wx1UcU3exNqTpVr
// SIG // j0CeyE0+rr1ZxSDWwnitt2RbO9DAFXA0FOXFnqYcx/T/
// SIG // 9stIweKN827OqrSgMIIGcDCCBFigAwIBAgIKYQxSTAAA
// SIG // AAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
// SIG // VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
// SIG // B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
// SIG // b3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3Qg
// SIG // Q2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAw
// SIG // NzA2MjA0MDE3WhcNMjUwNzA2MjA1MDE3WjB+MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQSAyMDEwMIIBIjANBgkqhkiG
// SIG // 9w0BAQEFAAOCAQ8AMIIBCgKCAQEA6Q5kUHlntcTj/QkA
// SIG // TJ6UrPdWaOpE2M/FWE+ppXZ8bUW60zmStKQe+fllguQX
// SIG // 0o/9RJwI6GWTzixVhL99COMuK6hBKxi3oktuSUxrFQfe
// SIG // 0dLCiR5xlM21f0u0rwjYzIjWaxeUOpPOJj/s5v40mFfV
// SIG // HV1J9rIqLtWFu1k/+JC0K4N0yiuzO0bj8EZJwRdmVMkc
// SIG // vR3EVWJXcvhnuSUgNN5dpqWVXqsogM3Vsp7lA7Vj07IU
// SIG // yMHIiiYKWX8H7P8O7YASNUwSpr5SW/Wm2uCLC0h31oVH
// SIG // 1RC5xuiq7otqLQVcYMa0KlucIxxfReMaFB5vN8sZM4Bq
// SIG // iU2jamZjeJPVMM+VHwIDAQABo4IB4zCCAd8wEAYJKwYB
// SIG // BAGCNxUBBAMCAQAwHQYDVR0OBBYEFOb8X3u7IgBY5HJO
// SIG // tfQhdCMy5u+sMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIA
// SIG // QwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/
// SIG // MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjE
// SIG // MFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
// SIG // b0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcB
// SIG // AQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0
// SIG // XzIwMTAtMDYtMjMuY3J0MIGdBgNVHSAEgZUwgZIwgY8G
// SIG // CSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9k
// SIG // ZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUA
// SIG // ZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBt
// SIG // AGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAGnTv
// SIG // V08pe8QWhXi4UNMi/AmdrIKX+DT/KiyXlRLl5L/Pv5PI
// SIG // 4zSp24G43B4AvtI1b6/lf3mVd+UC1PHr2M1OHhthosJa
// SIG // IxrwjKhiUUVnCOM/PB6T+DCFF8g5QKbXDrMhKeWloWmM
// SIG // IpPMdJjnoUdD8lOswA8waX/+0iUgbW9h098H1dlyACxp
// SIG // hnY9UdumOUjJN2FtB91TGcun1mHCv+KDqw/ga5uV1n0o
// SIG // UbCJSlGkmmzItx9KGg5pqdfcwX7RSXCqtq27ckdjF/qm
// SIG // 1qKmhuyoEESbY7ayaYkGx0aGehg/6MUdIdV7+QIjLcVB
// SIG // y78dTMgW77Gcf/wiS0mKbhXjpn92W9FTeZGFndXS2z1z
// SIG // NfM8rlSyUkdqwKoTldKOEdqZZ14yjPs3hdHcdYWch8Za
// SIG // V4XCv90Nj4ybLeu07s8n07VeafqkFgQBpyRnc89NT7be
// SIG // BVaXevfpUk30dwVPhcbYC/GO7UIJ0Q124yNWeCImNr7K
// SIG // sYxuqh3khdpHM2KPpMmRM19xHkCvmGXJIuhCISWKHC1g
// SIG // 2TeJQYkqFg/XYTyUaGBS79ZHmaCAQO4VgXc+nOBTGBpQ
// SIG // HTiVmx5mMxMnORd4hzbOTsNfsvU9R1O24OXbC2E9KteS
// SIG // LM43Wj5AQjGkHxAIwlacvyRdUQKdannSF9PawZSOB3sl
// SIG // cUSrBmrm1MbfI5qWdcUxghXFMIIVwQIBATCBlTB+MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3Nv
// SIG // ZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEwAhMzAAABT+fG
// SIG // YslG9Kl/AAAAAAFPMA0GCWCGSAFlAwQCAQUAoIGwMBkG
// SIG // CSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
// SIG // gjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJ
// SIG // BDEiBCC3lGXVO2sa125249YcFGBi4/M0P6VT0aYXOW1o
// SIG // nwW8NTBEBgorBgEEAYI3AgEMMTYwNKAagBgAcAByAG8A
// SIG // YwBlAHMAcwBvAHIALgBqAHOhFoAUaHR0cDovL21pY3Jv
// SIG // c29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEAg0AgwHyD
// SIG // rn855OP4EkRjO77uodSPKwVy8WETo6inY6bkUx4jXmKy
// SIG // gdFZFiKGr9H5K1Q+r9Prc2m2rROfvkJkCPGpBBF3UoQh
// SIG // 6u2sG5r0UjY5pFahuej6gscCobY+D68a8De3QTiqtDRZ
// SIG // RMMZLgq24W8IFX7q8GAvbCZoV01uwBeDEUcytsV9ZZjo
// SIG // rfYzJx1ahImO+L2HM/OjPRvsg/FV/siTKwad8/9RPj37
// SIG // XbqBvS5PrTrI4kgQIYEx+D6QjH/2SAcM5YVzjCHMV7XZ
// SIG // TN7m3zF5aPyymCZOgD67EJOcpHTf+PhMCox8AAJnS6Nx
// SIG // YFPAoLZc6VfLS6Va4LRtlLLrx6GCE00wghNJBgorBgEE
// SIG // AYI3AwMBMYITOTCCEzUGCSqGSIb3DQEHAqCCEyYwghMi
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwggE9BgsqhkiG9w0B
// SIG // CRABBKCCASwEggEoMIIBJAIBAQYKKwYBBAGEWQoDATAx
// SIG // MA0GCWCGSAFlAwQCAQUABCAf/2qC5iFZWBvLKUYjuUCq
// SIG // bxQH9S43XUI1u5PFpWxUywIGWD9ByyqoGBMyMDE2MTIy
// SIG // ODAwNTgwMy42OTJaMAcCAQGAAgH0oIG5pIG2MIGzMQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScw
// SIG // JQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QkJFQy0zMENB
// SIG // LTJEQkUxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFNlcnZpY2Wggg7QMIIGcTCCBFmgAwIBAgIKYQmB
// SIG // KgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcN
// SIG // MTAwNzAxMjEzNjU1WhcNMjUwNzAxMjE0NjU1WjB8MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
// SIG // ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8
// SIG // E5f1+n9plGt0VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5m
// SIG // K1vwFVMnBDEfQRsalR3OCROOfGEwWbEwRA/xYIiEVEMM
// SIG // 1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEby
// SIG // WEeGMoQedGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC7
// SIG // 32H8RsEnHSRnEnIaIYqvS2SJUGKxXf13Hz3wV3WsvYpC
// SIG // TUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw
// SIG // 2k4GkbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN
// SIG // /LzAyURdXhacAQVPIk0CAwEAAaOCAeYwggHiMBAGCSsG
// SIG // AQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
// SIG // fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBi
// SIG // AEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB
// SIG // /zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoY
// SIG // xDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNS
// SIG // b29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1
// SIG // dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0gAQH/BIGVMIGS
// SIG // MIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0
// SIG // dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9D
// SIG // UFMvZGVmYXVsdC5odG0wQAYIKwYBBQUHAgIwNB4yIB0A
// SIG // TABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0
// SIG // AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIB
// SIG // AAfmiFEN4sbgmD+BcQM9naOhIW+z66bM9TG+zwXiqf76
// SIG // V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
// SIG // +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDI
// SIG // r79/xn/yN31aPxzymXlKkVIArzgPF/UveYFl2am1a+TH
// SIG // zvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon/VWv
// SIG // L/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jns
// SIG // GUpxY517IW3DnKOiPPp/fZZqkHimbdLhnPkd/DjYlPTG
// SIG // pQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/fmNZJQ96
// SIG // LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/Dh
// SIG // O3EJ3110mCIIYdqwUB5vvfHhAN/nMQekkzr3ZUd46Pio
// SIG // SKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0cs0d9LiFAR6A
// SIG // +xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1Qn
// SIG // IffIrE7aKLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0
// SIG // MkvfY3v1mYovG8chr1m1rtxEPJdQcdeh0sVV42neV8HR
// SIG // 3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc
// SIG // 1bN+NR4Iuto229Nfj950iEkSMIIE2jCCA8KgAwIBAgIT
// SIG // MwAAAKGl/bnup/yenQAAAAAAoTANBgkqhkiG9w0BAQsF
// SIG // ADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
// SIG // aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0x
// SIG // NjA5MDcxNzU2NDhaFw0xODA5MDcxNzU2NDhaMIGzMQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScw
// SIG // JQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QkJFQy0zMENB
// SIG // LTJEQkUxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFNlcnZpY2UwggEiMA0GCSqGSIb3DQEBAQUAA4IB
// SIG // DwAwggEKAoIBAQCb0AF55iKvenwbdR7zt2fWHd6n3gA0
// SIG // 2BL5B+oIex9NTcqFlHqrdmsqB0WNUSfTtl1GpSXGYhR5
// SIG // i0/M5iz17J9Neh00IjYquMPi7MVQ7dP9Q8Etv8Xw+s/M
// SIG // hJDroqaybVegj7lhcNRJzogvgy47gUqTtUlKxGXJloXk
// SIG // L/qs4thXHTP2vhDnwlIbE+D5FDaos5v02xXw9NJrfS24
// SIG // Vc4R6Vb/lOkhDruRV8ycFXlwzY6s0+OBmZjDDgff23PF
// SIG // zylj9T7sNxh6c/YkdbX8yTeUMFcH1aBAFU0LFrDm1Tdd
// SIG // PNjTq7yHl4d6VXNLYUPB8wmIkr7OuOWESjwWN5xBziCX
// SIG // gcgJAgMBAAGjggEbMIIBFzAdBgNVHQ4EFgQUq1y5gr5x
// SIG // KtWP4OiyKc/6G9O3+kgwHwYDVR0jBBgwFoAU1WM6XIox
// SIG // kPNDe3xGG8UzaFqFbVUwVgYDVR0fBE8wTTBLoEmgR4ZF
// SIG // aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwv
// SIG // cHJvZHVjdHMvTWljVGltU3RhUENBXzIwMTAtMDctMDEu
// SIG // Y3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+
// SIG // aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0
// SIG // cy9NaWNUaW1TdGFQQ0FfMjAxMC0wNy0wMS5jcnQwDAYD
// SIG // VR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQsFAAOCAQEAI2BMuAVU6WfJvjk8FTLD
// SIG // E8izAJB/FzOb/XYvXUwSs+iwJKL+svQfqLKOLk4QB1zo
// SIG // 9zSK7jQd6OFEhvge2949EpSHwZoPQ+Cb+hRhq7bqEuOq
// SIG // iGrXZXflB1vQUFPRxVrUKC8qSlF5H3k5KnfHYeUjfyoF
// SIG // 2iae1UC24l/cOhN05Tr9qvs/Avwr+fggUlsoyl2yICju
// SIG // HR70ioS8F1LqsxJxmiwdG04NeNHbkw0kXheISVQh/Nhc
// SIG // JtDpE+Fsyk6/B7g7+eGcL0YMZTqcRbAJp3NMLGu21xZj
// SIG // 4PxyOJmmBc0ykUGiXvq7160Oe4XL8w93O3gy00+WkRRI
// SIG // I8aKl5dYHf2lQaGCA3kwggJhAgEBMIHjoYG5pIG2MIGz
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BS
// SIG // MScwJQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QkJFQy0z
// SIG // MENBLTJEQkUxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFNlcnZpY2WiJQoBATAJBgUrDgMCGgUAAxUA
// SIG // gq6J9bTmPxZcILwX5MHDQM0562iggcIwgb+kgbwwgbkx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIx
// SIG // JzAlBgNVBAsTHm5DaXBoZXIgTlRTIEVTTjo1N0Y2LUMx
// SIG // RTAtNTU0QzErMCkGA1UEAxMiTWljcm9zb2Z0IFRpbWUg
// SIG // U291cmNlIE1hc3RlciBDbG9jazANBgkqhkiG9w0BAQUF
// SIG // AAIFANwNhT4wIhgPMjAxNjEyMjgwMDI2MzhaGA8yMDE2
// SIG // MTIyOTAwMjYzOFowdzA9BgorBgEEAYRZCgQBMS8wLTAK
// SIG // AgUA3A2FPgIBADAKAgEAAgIBFAIB/zAHAgEAAgIa/jAK
// SIG // AgUA3A7WvgIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgor
// SIG // BgEEAYRZCgMBoAowCAIBAAIDFuNgoQowCAIBAAIDB6Eg
// SIG // MA0GCSqGSIb3DQEBBQUAA4IBAQDNbzKPpGOELvF2MG+h
// SIG // dKXjDjhHYP4Qcpyf77Ah4+lKRn+YPAveRXRI8YfLT6Xk
// SIG // 0WgRul/roLtGW7yJ+bC/bKE6FHMPRKngSSC8U1T5ub5l
// SIG // 14GB0A/xUXv2ceSwzTwVThQRMg1p4VL9loD1QVCXUMcS
// SIG // 9XQoUVcZGGL/iEuQiX+3s4ZB7HHjA0NKufF3DR6PqI8z
// SIG // KqXfMKo+aHoAJvDzxxIYXVzMY/a4/uZg7NV9CI6JjMd7
// SIG // doNUY9ONJRT3SuzX6pqFRIeH6/KFkr6UJy3yXKrHG5Qi
// SIG // dAP56GI5ORTXUpN88J1BrucPfVKN97mAiJ9QElvbwObB
// SIG // QfP4czyynlR6zOdzMYIC9TCCAvECAQEwgZMwfDELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
// SIG // IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAChpf257qf8
// SIG // np0AAAAAAKEwDQYJYIZIAWUDBAIBBQCgggEyMBoGCSqG
// SIG // SIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
// SIG // CQQxIgQgl3+QddZLQkbnm58z3rsxPVJ6n5h3alpGO0mK
// SIG // zPoHuRUwgeIGCyqGSIb3DQEJEAIMMYHSMIHPMIHMMIGx
// SIG // BBSCron1tOY/FlwgvBfkwcNAzTnraDCBmDCBgKR+MHwx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
// SIG // c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAAoaX9
// SIG // ue6n/J6dAAAAAAChMBYEFEtXhZ9d7pA2/2JCeJ6TcrdD
// SIG // /2oxMA0GCSqGSIb3DQEBCwUABIIBAD6u1pKdTeJnIRwS
// SIG // qUmTtNwip6YXJkIOGH6SUorzONNQrPxiDpJrSul7nscf
// SIG // ePY5lNZpdxmutpoP7XpV3Xsr4YAlOm+DGaXAHhBSSxyR
// SIG // FMu/hP6k9TDJuX+hLsC9BWisQV9DTCU8h2UFSkZSl2DS
// SIG // RRva+KxyBstZkFW1FmQeeSLoL64pDn86VGE08+NAAYq6
// SIG // OoQZlpCN+tPH0iEUZ5mvmCUVoXdISnbk2okagYMGLhFt
// SIG // M76u5EhB0snFqWYrlIDoroYyz0F0pRdruShGd3oRRC3u
// SIG // Xb9dLRWH0GcqH2JsMx1++qcOEEmZMPcMNnB8wLBiNiiC
// SIG // tLZrNkOSscrwS48BqLk=
// SIG // End signature block
